/*************************************************************************\
*                  Copyright (C) Michael Kerrisk, 2026.                   *
*                                                                         *
* This program is free software. You may use, modify, and redistribute it *
* under the terms of the GNU Lesser General Public License as published   *
* by the Free Software Foundation, either version 3 or (at your option)   *
* any later version. This program is distributed without any warranty.    *
* See the files COPYING.lgpl-v3 and COPYING.gpl-v3 for details.           *
\*************************************************************************/

/* Supplementary program for Chapter 61 */

#include <string.h>
#include <errno.h>
#include "scm_functions.h"

/* Send the file descriptor 'fd' over the connected UNIX domain socket
   'sockfd'. Returns 0 on success, or -1 on error. */

int
sendfd(int sockfd, int fd, void *data, size_t dataLen)
{
    /* Allocate a char array of suitable size to hold the ancillary data.
       However, since this buffer is in reality a 'struct cmsghdr', use a
       union to ensure that it is aligned as required for that structure.
       Alternatively, we could allocate the buffer using malloc(), which
       returns a buffer that satisfies the strictest alignment requirements
       of any type. However, if we employ that approach, we must ensure
       that we free() the buffer on all return paths from this function. */
    union {
        char   buf[CMSG_SPACE(sizeof(int))];
                        /* Space to hold an 'int' (a file descriptor) */
        struct cmsghdr align;
    } controlMsg;

    /* The 'msg_name' field can be used to specify the address of the
       destination socket when sending a datagram. However, we do not need
       to use this field because we presume that 'sockfd' is a connected
       socket. */

    struct msghdr msgh;
    msgh.msg_name = NULL;
    msgh.msg_namelen = 0;

    /* On Linux, we must transmit at least one byte of real data in order to
       send ancillary data. We transmit the data specified by 'data' and
       'dataLen', or some dummy data if 'data' is NULL. */

    struct iovec iov;
    msgh.msg_iov = &iov;
    msgh.msg_iovlen = 1;

    int dummyData = 12345;

    if (data != NULL) {
        iov.iov_base = data;
        iov.iov_len = dataLen;
    } else {
        iov.iov_base = &dummyData;
        iov.iov_len = sizeof(dummyData);
    }

    /* Set 'msghdr' fields that describe ancillary data. */

    msgh.msg_control = controlMsg.buf;
    msgh.msg_controllen = sizeof(controlMsg.buf);

    /* Set up ancillary data describing file descriptor to send. */

    struct cmsghdr *cmsgp;
    cmsgp = CMSG_FIRSTHDR(&msgh);
    cmsgp->cmsg_level = SOL_SOCKET;
    cmsgp->cmsg_type = SCM_RIGHTS;
    cmsgp->cmsg_len = CMSG_LEN(sizeof(int));
    memcpy(CMSG_DATA(cmsgp), &fd, sizeof(int));

    /* Send real plus ancillary data. */

    if (sendmsg(sockfd, &msgh, 0) == -1)
        return -1;

    return 0;
}

/* Receive a file descriptor on a connected UNIX domain socket. Returns the
   received file descriptor on success, or -1 on error. */

int
recvfd(int sockfd, void *data, size_t dataLen)
{
    /* Allocate a char buffer for the ancillary data. See the comments in
       sendfd(). */
    union {
        char   buf[CMSG_SPACE(sizeof(int))];
        struct cmsghdr align;
    } controlMsg;
    struct cmsghdr *cmsgp;

    /* The 'msg_name' field can be used to obtain the address of the
       sending socket. However, we do not need this information. */

    struct msghdr msgh;
    msgh.msg_name = NULL;
    msgh.msg_namelen = 0;

    /* Specify buffer for receiving real data. */

    struct iovec iov;
    msgh.msg_iov = &iov;
    msgh.msg_iovlen = 1;

    int dummyData;
    if (data != NULL) {
        iov.iov_base = data;
        iov.iov_len = dataLen;
    } else {
        iov.iov_base = &dummyData;
        iov.iov_len = sizeof(dummyData);
    }

    /* Set 'msghdr' fields that describe ancillary data. */

    msgh.msg_control = controlMsg.buf;
    msgh.msg_controllen = sizeof(controlMsg.buf);

    /* Receive real plus ancillary data; content of real data is ignored. */

    ssize_t nr = recvmsg(sockfd, &msgh, 0);
    if (nr == -1)
        return -1;

    cmsgp = CMSG_FIRSTHDR(&msgh);

    /* Check the validity of the 'cmsghdr'. */

    if (cmsgp == NULL ||
            cmsgp->cmsg_len != CMSG_LEN(sizeof(int)) ||
            cmsgp->cmsg_level != SOL_SOCKET ||
            cmsgp->cmsg_type != SCM_RIGHTS) {
        errno = EINVAL;
        return -1;
    }

    /* Return the received file descriptor to our caller. */

    int fd;
    memcpy(&fd, CMSG_DATA(cmsgp), sizeof(int));
    return fd;
}
